//GEMINI-pro DATASET v1.0 Category: Image to ASCII art converter ; Style: thoughtful
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

// Define the maximum width and height of the image
#define MAX_WIDTH 1024
#define MAX_HEIGHT 1024

// Define the number of ASCII characters to use
#define NUM_ASCII_CHARS 70

// Define the ASCII character map
char ascii_map[] = {
    ' ', '.', ',', '-', '=', '+', '*', '#', '%', '@',
    '&', '$', '!', ':', ';', '?', '(', ')', '{', '}',
    '[', ']', '<', '>', '\\', '/', '^', '_', '`', '~',
    '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
    'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J',
    'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T',
    'U', 'V', 'W', 'X', 'Y', 'Z', 'a', 'b', 'c', 'd',
    'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
    'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x',
    'y', 'z'
};

// Load an image from a file
unsigned char *load_image(const char *filename, int *width, int *height) {
    // Open the file
    FILE *fp = fopen(filename, "rb");
    if (!fp) {
        fprintf(stderr, "Error opening file: %s\n", filename);
        return NULL;
    }

    // Read the header information
    unsigned char header[54];
    fread(header, sizeof(header), 1, fp);

    // Check if the file is a valid BMP file
    if (header[0] != 'B' || header[1] != 'M') {
        fprintf(stderr, "Error: file is not a valid BMP file\n");
        fclose(fp);
        return NULL;
    }

    // Get the width and height of the image
    *width = *(int *)&header[18];
    *height = *(int *)&header[22];

    // Allocate memory for the image data
    unsigned char *data = malloc(*width * *height * 3);
    if (!data) {
        fprintf(stderr, "Error allocating memory for image data\n");
        fclose(fp);
        return NULL;
    }

    // Read the image data
    fread(data, *width * *height * 3, 1, fp);

    // Close the file
    fclose(fp);

    return data;
}

// Convert an image to ASCII art
char *image_to_ascii(unsigned char *data, int width, int height) {
    // Allocate memory for the ASCII art
    char *ascii_art = malloc(width * height);
    if (!ascii_art) {
        fprintf(stderr, "Error allocating memory for ASCII art\n");
        return NULL;
    }

    // Convert the image data to ASCII art
    for (int i = 0; i < width * height; i++) {
        // Calculate the average of the RGB values
        int r = data[i * 3 + 0];
        int g = data[i * 3 + 1];
        int b = data[i * 3 + 2];
        int avg = (r + g + b) / 3;

        // Map the average value to an ASCII character
        ascii_art[i] = ascii_map[avg * NUM_ASCII_CHARS / 256];
    }

    return ascii_art;
}

// Save the ASCII art to a file
void save_ascii_art(const char *filename, char *ascii_art, int width, int height) {
    // Open the file
    FILE *fp = fopen(filename, "wb");
    if (!fp) {
        fprintf(stderr, "Error opening file: %s\n", filename);
        return;
    }

    // Write the ASCII art to the file
    fwrite(ascii_art, width * height, 1, fp);

    // Close the file
    fclose(fp);
}

// Main function
int main(int argc, char *argv[]) {
    // Check if the user specified an input file
    if (argc < 2) {
        fprintf(stderr, "Usage: %s <input_file>\n", argv[0]);
        return EXIT_FAILURE;
    }

    // Load the image from the file
    int width, height;
    unsigned char *data = load_image(argv[1], &width, &height);
    if (!data) {
        return EXIT_FAILURE;
    }

    // Convert the image to ASCII art
    char *ascii_art = image_to_ascii(data, width, height);
    if (!ascii_art) {
        return EXIT_FAILURE;
    }

    // Save the ASCII art to a file
    save_ascii_art("output.txt", ascii_art, width, height);

    // Free the allocated memory
    free(data);
    free(ascii_art);

    return EXIT_SUCCESS;
}