//GEMINI-pro DATASET v1.0 Category: Classical Circuit Simulator ; Style: detailed
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define MAX_NODES 100
#define MAX_INPUTS 10
#define MAX_OUTPUTS 10

typedef struct node {
    char *name;
    int value;
    int num_inputs;
    int *inputs;
    char *function;
} node;

typedef struct circuit {
    int num_nodes;
    node *nodes;
    int num_inputs;
    int *inputs;
    int num_outputs;
    int *outputs;
} circuit;

circuit *create_circuit(char *filename) {
    FILE *fp;
    char line[1024];
    char *token;
    int num_nodes, num_inputs, num_outputs;
    circuit *ckt;

    // Open the circuit file.
    fp = fopen(filename, "r");
    if (fp == NULL) {
        fprintf(stderr, "Error: unable to open circuit file %s\n", filename);
        return NULL;
    }

    // Read the first line of the file, which contains the number of nodes, inputs, and outputs.
    fgets(line, sizeof(line), fp);
    sscanf(line, "%d %d %d", &num_nodes, &num_inputs, &num_outputs);

    // Allocate memory for the circuit.
    ckt = malloc(sizeof(circuit));
    if (ckt == NULL) {
        fprintf(stderr, "Error: unable to allocate memory for circuit\n");
        fclose(fp);
        return NULL;
    }

    // Initialize the circuit.
    ckt->num_nodes = num_nodes;
    ckt->nodes = malloc(num_nodes * sizeof(node));
    if (ckt->nodes == NULL) {
        fprintf(stderr, "Error: unable to allocate memory for nodes\n");
        free(ckt);
        fclose(fp);
        return NULL;
    }
    ckt->num_inputs = num_inputs;
    ckt->inputs = malloc(num_inputs * sizeof(int));
    if (ckt->inputs == NULL) {
        fprintf(stderr, "Error: unable to allocate memory for inputs\n");
        free(ckt->nodes);
        free(ckt);
        fclose(fp);
        return NULL;
    }
    ckt->num_outputs = num_outputs;
    ckt->outputs = malloc(num_outputs * sizeof(int));
    if (ckt->outputs == NULL) {
        fprintf(stderr, "Error: unable to allocate memory for outputs\n");
        free(ckt->nodes);
        free(ckt->inputs);
        free(ckt);
        fclose(fp);
        return NULL;
    }

    // Read the rest of the file, which contains the nodes.
    for (int i = 0; i < num_nodes; i++) {
        // Read the next line of the file.
        fgets(line, sizeof(line), fp);

        // Parse the line.
        token = strtok(line, " ");
        ckt->nodes[i].name = strdup(token);
        token = strtok(NULL, " ");
        ckt->nodes[i].value = atoi(token);
        token = strtok(NULL, " ");
        ckt->nodes[i].num_inputs = atoi(token);
        ckt->nodes[i].inputs = malloc(ckt->nodes[i].num_inputs * sizeof(int));
        if (ckt->nodes[i].inputs == NULL) {
            fprintf(stderr, "Error: unable to allocate memory for inputs\n");
            for (int j = 0; j < i; j++) {
                free(ckt->nodes[j].name);
                free(ckt->nodes[j].inputs);
            }
            free(ckt->nodes);
            free(ckt->inputs);
            free(ckt->outputs);
            free(ckt);
            fclose(fp);
            return NULL;
        }
        for (int j = 0; j < ckt->nodes[i].num_inputs; j++) {
            token = strtok(NULL, " ");
            ckt->nodes[i].inputs[j] = atoi(token);
        }
        token = strtok(NULL, " ");
        ckt->nodes[i].function = strdup(token);
    }

    // Close the circuit file.
    fclose(fp);

    // Return the circuit.
    return ckt;
}

void destroy_circuit(circuit *ckt) {
    for (int i = 0; i < ckt->num_nodes; i++) {
        free(ckt->nodes[i].name);
        free(ckt->nodes[i].inputs);
        free(ckt->nodes[i].function);
    }
    free(ckt->nodes);
    free(ckt->inputs);
    free(ckt->outputs);
    free(ckt);
}

void simulate_circuit(circuit *ckt) {
    // Initialize the inputs to 0.
    for (int i = 0; i < ckt->num_inputs; i++) {
        ckt->inputs[i] = 0;
    }

    // Simulate the circuit.
    for (int i = 0; i < ckt->num_nodes; i++) {
        // Get the node's function.
        char *function = ckt->nodes[i].function;

        // Evaluate the node's function.
        int value = 0;
        if (strcmp(function, "AND") == 0) {
            for (int j = 0; j < ckt->nodes[i].num_inputs; j++) {
                value &= ckt->nodes[ckt->nodes[i].inputs[j]].value;
            }
        } else if (strcmp(function, "OR") == 0) {
            for (int j = 0; j < ckt->nodes[i].num_inputs; j++) {
                value |= ckt->nodes[ckt->nodes[i].inputs[j]].value;
            }
        } else if (strcmp(function, "XOR") == 0) {
            for (int j = 0; j < ckt->nodes[i].num_inputs; j++) {
                value ^= ckt->nodes[ckt->nodes[i].inputs[j]].value;
            }
        } else if (strcmp(function, "NOT") == 0) {
            value = !ckt->nodes[ckt->nodes[i].inputs[0]].value;
        }

        // Set the node's value.
        ckt->nodes[i].value = value;
    }

    // Print the outputs.
    for (int i = 0; i < ckt->num_outputs; i++) {
        printf("%d ", ckt->outputs[i]);
    }
    printf("\n");
}

int main(int argc, char **argv) {
    if (argc != 2) {
        fprintf(stderr, "Usage: %s <circuit file>\n", argv[0]);
        return 1;
    }

    // Create the circuit.
    circuit *ckt = create_circuit(argv[1]);
    if (ckt == NULL) {
        return 1;
    }

    // Simulate the circuit.
    simulate_circuit(ckt);

    // Destroy the circuit.
    destroy_circuit(ckt);

    return 0;
}