//GPT-4o-mini DATASET v1.0 Category: Building a JSON Parser ; Style: interoperable
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

typedef struct JsonValue {
    enum { JSON_NULL, JSON_BOOLEAN, JSON_NUMBER, JSON_STRING, JSON_ARRAY, JSON_OBJECT } type;
    union {
        int boolean;
        double number;
        char *string;
        struct JsonValue *array; // Will be implemented in array management
        struct {
            char **keys;
            struct JsonValue *values;
        } object; // For JSON objects
    } value;
    size_t size; // For arrays and objects
} JsonValue;

void trim_whitespace(const char **json) {
    while (isspace((unsigned char)**json)) (*json)++;
}

JsonValue parse_value(const char **json);

JsonValue parse_null(const char **json) {
    if (strncmp(*json, "null", 4) == 0) {
        *json += 4;
        JsonValue value;
        value.type = JSON_NULL;
        return value;
    }
    fprintf(stderr, "Error parsing null value\n");
    exit(EXIT_FAILURE);
}

JsonValue parse_boolean(const char **json) {
    JsonValue value;
    if (strncmp(*json, "true", 4) == 0) {
        value.type = JSON_BOOLEAN;
        value.value.boolean = 1;
        *json += 4;
    } else if (strncmp(*json, "false", 5) == 0) {
        value.type = JSON_BOOLEAN;
        value.value.boolean = 0;
        *json += 5;
    } else {
        fprintf(stderr, "Error parsing boolean value\n");
        exit(EXIT_FAILURE);
    }
    return value;
}

JsonValue parse_number(const char **json) {
    char *endptr;
    JsonValue value;
    value.type = JSON_NUMBER;
    value.value.number = strtod(*json, &endptr);
    if (endptr == *json) {
        fprintf(stderr, "Error parsing number value\n");
        exit(EXIT_FAILURE);
    }
    *json = endptr;
    return value;
}

JsonValue parse_string(const char **json) {
    JsonValue value;
    value.type = JSON_STRING;
    const char *start = ++(*json);
    while (**json != '"' && **json != '\0') {
        if (**json == '\\') (*json)++; // Skip escape character
        (*json)++;
    }
    size_t length = *json - start;
    if (**json == '\0') {
        fprintf(stderr, "Error parsing string value\n");
        exit(EXIT_FAILURE);
    }
    value.value.string = (char*)malloc(length + 1);
    strncpy(value.value.string, start, length);
    value.value.string[length] = '\0';
    (*json)++; // Skip closing quote
    return value;
}

JsonValue parse_array(const char **json) {
    JsonValue value;
    value.type = JSON_ARRAY;
    value.size = 0;
    value.value.array = NULL;

    while (**json != ']' && **json != '\0') {
        trim_whitespace(json);
        value.size++;
        value.value.array = realloc(value.value.array, sizeof(JsonValue) * value.size);
        value.value.array[value.size - 1] = parse_value(json);
        trim_whitespace(json);
        if (**json == ',') (*json)++;
    }
    
    if (**json == '\0') {
        fprintf(stderr, "Error parsing array\n");
        exit(EXIT_FAILURE);
    }
    (*json)++; // Skip closing bracket
    return value;
}

JsonValue parse_object(const char **json) {
    JsonValue value;
    value.type = JSON_OBJECT;
    value.size = 0;
    value.value.object.keys = NULL;
    value.value.object.values = NULL;

    while (**json != '}' && **json != '\0') {
        trim_whitespace(json);
        if (**json != '"') {
            fprintf(stderr, "Error parsing object key\n");
            exit(EXIT_FAILURE);
        }
        JsonValue key = parse_string(json);
        value.size++;
        value.value.object.keys = realloc(value.value.object.keys, sizeof(char*) * value.size);
        value.value.object.values = realloc(value.value.object.values, sizeof(JsonValue) * value.size);
        value.value.object.keys[value.size - 1] = key.value.string;

        trim_whitespace(json);
        if (**json != ':') {
            fprintf(stderr, "Error parsing object separator\n");
            exit(EXIT_FAILURE);
        }
        (*json)++;

        trim_whitespace(json);
        value.value.object.values[value.size - 1] = parse_value(json);
        trim_whitespace(json);

        if (**json == ',') (*json)++;
    }

    if (**json == '\0') {
        fprintf(stderr, "Error parsing object\n");
        exit(EXIT_FAILURE);
    }
    (*json)++; // Skip closing brace
    return value;
}

JsonValue parse_value(const char **json) {
    trim_whitespace(json);
    if (**json == 'n') return parse_null(json);
    if (**json == 't' || **json == 'f') return parse_boolean(json);
    if (isdigit(**json) || **json == '-') return parse_number(json);
    if (**json == '"') return parse_string(json);
    if (**json == '[') return parse_array(json);
    if (**json == '{') return parse_object(json);
    fprintf(stderr, "Error parsing value\n");
    exit(EXIT_FAILURE);
}

void free_value(JsonValue value) {
    switch (value.type) {
        case JSON_STRING:
            free(value.value.string);
            break;
        case JSON_ARRAY:
            for (size_t i = 0; i < value.size; i++) {
                free_value(value.value.array[i]);
            }
            free(value.value.array);
            break;
        case JSON_OBJECT:
            for (size_t i = 0; i < value.size; i++) {
                free(value.value.object.keys[i]);
                free_value(value.value.object.values[i]);
            }
            free(value.value.object.keys);
            free(value.value.object.values);
            break;
        default:
            break;
    }
}

int main(void) {
    const char *json = "{\"name\":\"John\",\"age\":30,\"is_student\":false,\"courses\":[\"Math\",\"CompSci\"],\"details\":null}";
    
    JsonValue result = parse_value(&json);
    
    // (The result is not printed here, but you can extend the functionality to display the parsed JSON structure)

    free_value(result);
    return 0;
}