//GEMINI-pro DATASET v1.0 Category: Building a HTTP Client ; Style: future-proof
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <curl/curl.h>

// Custom data structure to store HTTP response data
typedef struct {
    char *data;
    size_t size;
} HTTPResponseData;

// Callback function to receive and store HTTP response data
size_t write_callback(void *contents, size_t size, size_t nmemb, void *userp) {
    HTTPResponseData *response = (HTTPResponseData *) userp;
    size_t realsize = size * nmemb;

    // Reallocate memory for response data if necessary
    response->data = realloc(response->data, response->size + realsize + 1);
    if (response->data == NULL) {
        return 0;  // Error: out of memory
    }

    // Append response data to the buffer
    memcpy(response->data + response->size, contents, realsize);
    response->size += realsize;
    response->data[response->size] = '\0';

    return realsize;
}

// Send an HTTP GET request and retrieve the response
int send_http_get_request(const char *url, HTTPResponseData *response) {
    CURL *curl;
    CURLcode res;

    // Initialize the curl session
    curl = curl_easy_init();
    if (curl == NULL) {
        return -1;  // Error: curl initialization failed
    }

    // Set the URL to fetch
    curl_easy_setopt(curl, CURLOPT_URL, url);

    // Set the callback function to receive and store response data
    curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, write_callback);
    curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *) response);

    // Perform the HTTP GET request
    res = curl_easy_perform(curl);

    // Check for errors
    if (res != CURLE_OK) {
        fprintf(stderr, "curl_easy_perform() failed: %s\n", curl_easy_strerror(res));
        return -2;  // Error: curl request failed
    }

    // Cleanup the curl session
    curl_easy_cleanup(curl);

    return 0;  // Success
}

// Print the HTTP response data
void print_http_response(HTTPResponseData *response) {
    printf("HTTP Response:\n");
    printf("-----------------------------\n");
    printf("%s\n", response->data);
}

int main(int argc, char **argv) {
    // Check for argument count
    if (argc < 2) {
        printf("Usage: %s <URL>\n", argv[0]);
        return -1;  // Error: not enough arguments
    }

    // Allocate memory for HTTP response data
    HTTPResponseData response = {NULL, 0};

    // Send the HTTP GET request
    int status = send_http_get_request(argv[1], &response);
    if (status != 0) {
        return status;  // Error: HTTP request failed
    }

    // Print the HTTP response
    print_http_response(&response);

    // Free the allocated memory
    free(response.data);

    return 0;  // Success
}