File number 1:
#if HAVE_CONFIG_H
#include "config.h"
#endif
#include <string.h>
#include <stdlib.h>
#include "internal.h"
#include "asn1.h"
#include "cardctl.h"
#include "types.h"
#define LOAD_KEY_MODULUS		0x0080
#define LOAD_KEY_PUBLIC_EXPONENT	0x0081
#define LOAD_KEY_PRIME_P		0x0083
#define LOAD_KEY_PRIME_Q		0x0084
#define LOAD_KEY_DP1			0x0085
#define LOAD_KEY_DQ1			0x0086
#define LOAD_KEY_INVQ			0x0087
#define LOAD_KEY_EC_PUBLIC		0x1086
#define LOAD_KEY_EC_PRIVATE		0x1087
#define LOAD_KEY_SYMMETRIC		0x20a0
#define MYEID_CARD_NAME_MAX_LEN		100
#define MYEID_CARD_CAP_RSA		0x01
#define MYEID_CARD_CAP_3DES		0x02
#define MYEID_CARD_CAP_AES		0x04
#define MYEID_CARD_CAP_ECC		0x08
#define MYEID_CARD_CAP_GRIDPIN		0x10
#define MYEID_CARD_CAP_PIV_EMU		0x20
#define MYEID_MAX_APDU_DATA_LEN		0xFF
#define MYEID_MAX_RSA_KEY_LEN		4096
#define MYEID_MAX_EXT_APDU_BUFFER_SIZE	(MYEID_MAX_RSA_KEY_LEN/8+16)
static const char *myeid_card_name = "MyEID";
static const char *oseid_card_name = "OsEID";
static char card_name_buf[MYEID_CARD_NAME_MAX_LEN];
static struct sc_card_operations myeid_ops;
static struct sc_card_driver myeid_drv = {
	"MyEID cards with PKCS#15 applet",
	"myeid",
	&myeid_ops,
	NULL,
	0,
	NULL
};
typedef struct myeid_private_data {
	int card_state;
	unsigned short change_counter;
	unsigned char cap_chaining;
	const struct sc_security_env* sec_env;
	int disable_hw_pkcs1_padding;
	uint8_t sym_crypt_buffer[16];
	uint8_t sym_plain_buffer[16];
	uint8_t sym_crypt_buffer_len;
	uint8_t sym_plain_buffer_len;
	unsigned int algorithm, algorithm_flags;
} myeid_private_data_t;
typedef struct myeid_card_caps {
	unsigned char card_caps_ver;
	unsigned short card_supported_features;
	unsigned short max_rsa_key_length;
	unsigned short max_des_key_length;
	unsigned short max_aes_key_length;
	unsigned short max_ecc_key_length;
} myeid_card_caps_t;
static struct myeid_supported_ec_curves {
	char *curve_name;
	struct sc_object_id curve_oid;
	size_t size;
} ec_curves[] = {
	{"secp192r1", {{1, 2, 840, 10045, 3, 1, 1, -1}},192},
	{"secp256r1", {{1, 2, 840, 10045, 3, 1, 7, -1}},256},
	{"secp384r1", {{1, 3, 132, 0, 34, -1}},		384},
	{"secp521r1", {{1, 3, 132, 0, 35, -1}},		521},
	{NULL, {{-1}}, 0},
};
static int myeid_get_info(struct sc_card *card, u8 *rbuf, size_t buflen);
static int myeid_get_card_caps(struct sc_card *card, myeid_card_caps_t* card_caps);
static int
myeid_select_aid(struct sc_card *card, struct sc_aid *aid, unsigned char *out, size_t *out_len)
{
	struct sc_apdu apdu;
	unsigned char apdu_resp[SC_MAX_APDU_BUFFER_SIZE];
	int rv;
	sc_format_apdu(card, &apdu, SC_APDU_CASE_4_SHORT, 0xA4, 0x04, 0x00);
	apdu.lc = aid->len;
	apdu.data = aid->value;
	apdu.datalen = aid->len;
	apdu.resplen = sizeof(apdu_resp);
	apdu.resp = apdu_resp;
	rv = sc_transmit_apdu(card, &apdu);
	LOG_TEST_RET(card->ctx, rv, "APDU transmit failed");
	rv = sc_check_sw(card, apdu.sw1, apdu.sw2);
	LOG_TEST_RET(card->ctx, rv, "Cannot select AID");
	if (*out_len > 0) {
		if (*out_len < apdu.resplen)
			LOG_TEST_RET(card->ctx, SC_ERROR_BUFFER_TOO_SMALL, "Cannot select AID - response buffer too small.");
		if (out == NULL)
			LOG_TEST_RET(card->ctx, SC_ERROR_INVALID_ARGUMENTS, "Cannot select AID - invalid arguments.");
		memcpy(out, apdu.resp, apdu.resplen);
		*out_len = apdu.resplen;
	}
	return SC_SUCCESS;
}
static int myeid_init(struct sc_card *card)
{
	unsigned long flags = 0, ext_flags = 0;
	myeid_private_data_t *priv;
	u8 appletInfo[20];
	size_t appletInfoLen;
	myeid_card_caps_t card_caps;
	size_t resp_len = 0;
	static struct sc_aid myeid_aid = { "\xA0\x00\x00\x00\x63\x50\x4B\x43\x53\x2D\x31\x35", 0x0C };
	int rv = 0;
	void *old_drv_data = card->drv_data;
	LOG_FUNC_CALLED(card->ctx);
	switch (card->type) {
	case SC_CARD_TYPE_MYEID_OSEID:
		card->name = oseid_card_name;
		break;
	case SC_CARD_TYPE_MYEID_GENERIC:
		card->name = myeid_card_name;
		break;
	default:
		LOG_FUNC_RETURN(card->ctx, SC_ERROR_INVALID_CARD);
	}
	priv = calloc(1, sizeof(myeid_private_data_t));
	if (!priv)
		LOG_FUNC_RETURN(card->ctx, SC_ERROR_OUT_OF_MEMORY);
	rv = myeid_load_options (card->ctx, priv);
	LOG_TEST_GOTO_ERR(card->ctx, rv, "Unable to read options from opensc.conf");
	priv->card_state = SC_FILE_STATUS_CREATION;
	card->drv_data = priv;
	rv = myeid_select_aid(card, &myeid_aid, NULL, &resp_len);
	LOG_TEST_GOTO_ERR(card->ctx, rv, "Failed to select MyEID applet.");
	appletInfoLen = 20;
	if (0 > myeid_get_info(card, appletInfo, appletInfoLen))
		LOG_TEST_GOTO_ERR(card->ctx, SC_ERROR_INVALID_CARD, "Failed to get MyEID applet information.");
	priv->change_counter = appletInfo[19] | appletInfo[18] << 8;
	memset(&card_caps, 0, sizeof(myeid_card_caps_t));
	card_caps.max_ecc_key_length = 256;
	card_caps.max_rsa_key_length = 2048;
	if (card->version.fw_major >= 40) {
	    if (myeid_get_card_caps(card, &card_caps) != SC_SUCCESS) {
			sc_log(card->ctx, "Failed to get card capabilities. Using default max ECC key length 256.");
	    }
	}
	flags = SC_ALGORITHM_RSA_RAW | SC_ALGORITHM_ONBOARD_KEY_GEN;
	if (priv->disable_hw_pkcs1_padding == 0)
		flags |= SC_ALGORITHM_RSA_PAD_PKCS1;
	flags |= SC_ALGORITHM_RSA_HASH_NONE;
	_sc_card_add_rsa_alg(card,  512, flags, 0);
	_sc_card_add_rsa_alg(card,  768, flags, 0);
	_sc_card_add_rsa_alg(card, 1024, flags, 0);
	_sc_card_add_rsa_alg(card, 1536, flags, 0);
	_sc_card_add_rsa_alg(card, 2048, flags, 0);
	if (card_caps.card_supported_features & MYEID_CARD_CAP_RSA) {
		if (card_caps.max_rsa_key_length >= 3072)
			_sc_card_add_rsa_alg(card, 3072, flags, 0);
		if (card_caps.max_rsa_key_length >= 4096)
			_sc_card_add_rsa_alg(card, 4096, flags, 0);
	}
	if (card->version.fw_major >= 35) {
		int i;
		flags = SC_ALGORITHM_ECDSA_RAW | SC_ALGORITHM_ECDH_CDH_RAW | SC_ALGORITHM_ONBOARD_KEY_GEN;
		flags |= SC_ALGORITHM_ECDSA_HASH_NONE;
		ext_flags = SC_ALGORITHM_EXT_EC_NAMEDCURVE | SC_ALGORITHM_EXT_EC_UNCOMPRESES;
		for (i=0; ec_curves[i].curve_name != NULL; i++) {
			if (card_caps.max_ecc_key_length >= ec_curves[i].size)
				_sc_card_add_ec_alg(card, ec_curves[i].size, flags, ext_flags, &ec_curves[i].curve_oid);
		}
	}
	flags = 0;
	if (card_caps.card_supported_features & MYEID_CARD_CAP_3DES) {
		if (card_caps.max_des_key_length >= 64)
			_sc_card_add_symmetric_alg(card, SC_ALGORITHM_DES, 64, flags);
		if (card_caps.max_des_key_length >= 128)
			_sc_card_add_symmetric_alg(card, SC_ALGORITHM_3DES, 128, flags);
		if (card_caps.max_des_key_length >= 192)
			_sc_card_add_symmetric_alg(card, SC_ALGORITHM_3DES, 192, flags);
	}
	if (card_caps.card_supported_features & MYEID_CARD_CAP_AES) {
		if (card_caps.max_aes_key_length >= 128)
			_sc_card_add_symmetric_alg(card, SC_ALGORITHM_AES, 128, flags);
		if (card_caps.max_aes_key_length >= 256)
			_sc_card_add_symmetric_alg(card, SC_ALGORITHM_AES, 256, flags);
	}
	card->caps |= SC_CARD_CAP_RNG | SC_CARD_CAP_ISO7816_PIN_INFO;
	if ((card->version.fw_major == 40 && card->version.fw_minor >= 10 )
		|| card->version.fw_major >= 41)
		card->caps |= SC_CARD_CAP_WRAP_KEY | SC_CARD_CAP_UNWRAP_KEY
			   | SC_CARD_CAP_ONCARD_SESSION_OBJECTS;
	if (card->version.fw_major >= 45)
		priv->cap_chaining = 1;
	if (card->version.fw_major >= 40)
		card->max_recv_size = 256;
	else
		card->max_recv_size = 255;
	card->max_send_size = 255;
	rv = SC_SUCCESS;
err:
	if (rv < 0) {
		free(priv);
		card->drv_data = old_drv_data;
	}
	LOG_FUNC_RETURN(card->ctx, rv);
}
static const struct sc_card_operations *iso_ops = NULL;
static int myeid_process_fci(struct sc_card *card, struct sc_file *file,
		const u8 *buf, size_t buflen)
{
	myeid_private_data_t *priv = (myeid_private_data_t *) card->drv_data;
	size_t taglen = 0;
	const u8 *tag = NULL;
	int r;
	LOG_FUNC_CALLED(card->ctx);
	r = iso_ops->process_fci(card, file, buf, buflen);
	if (r < 0)
	 LOG_FUNC_RETURN(card->ctx, r);
	if(file->type == SC_FILE_EF_UNKNOWN)
	{
		tag = sc_asn1_find_tag(NULL, buf, buflen, 0x82, &taglen);
		if (tag != NULL && taglen > 0 && *tag == 17)
		{
			file->type = SC_FILE_TYPE_INTERNAL_EF;
		}
	}
	if(file->sec_attr_len >= 3)
	{
		sc_log(card->ctx, "id (%X) sec_attr (%X %X %X)", file->id,
			file->sec_attr[0],file->sec_attr[1],file->sec_attr[2]);
	}
	priv->card_state = file->status;
	switch (file->status) {
		case SC_FILE_STATUS_CREATION:
			file->acl_inactive = 1;
			sc_log(card->ctx, "File id (%X) status SC_FILE_STATUS_CREATION", file->id);
			break;
		case SC_FILE_STATUS_ACTIVATED:
			sc_log(card->ctx, "File id (%X) status SC_FILE_STATUS_ACTIVATED", file->id);
			break;
		default:
			sc_log(card->ctx, "File id (%X) unusual status (0x%X)", file->id, file->status);
	}
	LOG_FUNC_RETURN(card->ctx, 0);
}
static int encode_file_structure(sc_card_t *card, const sc_file_t *file,
		u8 *buf, size_t *outlen)
{
	const sc_acl_entry_t *read, *update, *delete, *generate;
	size_t i;
	LOG_FUNC_CALLED(card->ctx);
	if (!buf || !outlen || *outlen < 45)
		LOG_FUNC_RETURN(card->ctx, SC_ERROR_INTERNAL);
	memset(buf, 0x0, *outlen);
	buf[0] = 0x62;
	buf[1] = 0x17;
	buf[2] = (SC_FILE_TYPE_WORKING_EF == file->type ? 0x80 : 0x81);
	buf[3] = 0x02;
	buf[4] = (file->size >> 8) & 0xFF;
	buf[5] = file->size & 0xFF;
	buf[6] = 0x82;
	buf[7] = 0x01;
	buf[8] = 0x01;
	buf[9]  = 0x83;
	buf[10] = 0x02;
	buf[11] = (file->id >> 8) & 0xFF;
	buf[12] = file->id & 0xFF;
	buf[13] = 0x86;
	buf[14] = 0x03;
	buf[15] = 0xFF;
	buf[16] = 0xFF;
	buf[17] = 0xFF;
	if (file->sec_attr_len == 3 && file->sec_attr)   {
		buf[15] = file->sec_attr[0];
		buf[16] = file->sec_attr[1];
		buf[17] = file->sec_attr[2];
		sc_log(card->ctx, "id (%X), sec_attr %X %X %X", file->id,
				file->sec_attr[0],file->sec_attr[1],file->sec_attr[2]);
	}
	else   {
		delete = sc_file_get_acl_entry(file, SC_AC_OP_DELETE);
		sc_log(card->ctx, "id (%X), type (%X)", file->id, file->type);
		switch (file->type) {
		case SC_FILE_TYPE_WORKING_EF:
			read = sc_file_get_acl_entry(file, SC_AC_OP_READ);
			update = sc_file_get_acl_entry(file, SC_AC_OP_UPDATE);
			buf[15] = (acl_to_byte(read) << 4) | acl_to_byte(update);
			buf[16] = (acl_to_byte(delete)<< 4) | 0x0F;
			break;
		case SC_FILE_TYPE_INTERNAL_EF:
			read = sc_file_get_acl_entry(file, SC_AC_OP_CRYPTO);
			update = sc_file_get_acl_entry(file, SC_AC_OP_UPDATE);
			generate = sc_file_get_acl_entry(file, SC_AC_OP_GENERATE);
			buf[15] = (acl_to_byte(read) << 4) | acl_to_byte(update);
			buf[16] = (acl_to_byte(delete)<< 4) | acl_to_byte(generate);
			break;
		case SC_FILE_TYPE_DF:
			update = sc_file_get_acl_entry(file, SC_AC_OP_CREATE);
			buf[15] = (acl_to_byte(update) << 4) | acl_to_byte(update);
			buf[16] = (acl_to_byte(delete) << 4) | 0x0F;
			break;
		default:
			break;
		}
	}
	buf[18] = 0x85;
	buf[19] = 0x02;
	if (file->prop_attr_len == 2 && file->prop_attr != NULL)
	    memcpy(&buf[20], file->prop_attr, 2);
	else
	{
		buf[20] = 0x00;
		buf[21] = 0x00;
	}
	buf[22] = 0x8A;
	buf[23] = 0x01;
	buf[24] = 0x0; 
	switch (file->type)
	{
	case SC_FILE_TYPE_WORKING_EF:
		break;
	case SC_FILE_TYPE_INTERNAL_EF:
		buf[8] = file->ef_structure; 
		break;
	case SC_FILE_TYPE_DF:
		buf[8] = 0x38;
		if(file->namelen > 0 && file->namelen <= 16)
		{
			buf[25] = 0x84;
			buf[26] = (u8)file->namelen;
			for(i=0;i < file->namelen;i++)
				buf[i + 27] = file->name[i];
			buf[1] = 27 + file->namelen;
		}
		break;
	default:
		sc_log(card->ctx, "Unknown file type\n");
		return SC_ERROR_INVALID_ARGUMENTS;
	}
	*outlen = buf[1]+2;
	LOG_FUNC_RETURN(card->ctx, SC_SUCCESS);
}
#define IS_SYMETRIC_CRYPT(x) ((x) == SC_SEC_OPERATION_ENCRYPT_SYM || (x) == SC_SEC_OPERATION_DECRYPT_SYM)
static int myeid_set_security_env_rsa(sc_card_t *card, const sc_security_env_t *env,
		int se_num)
{
	sc_apdu_t apdu;
	u8 sbuf[SC_MAX_APDU_BUFFER_SIZE];
	u8 *p;
	int r;
	size_t i;
	sc_path_t *target_file;
	assert(card != NULL && env != NULL);
	LOG_FUNC_CALLED(card->ctx);
	if (env->flags & SC_SEC_ENV_KEY_REF_SYMMETRIC)
	{
		sc_log(card->ctx, "symmetric keyref not supported.\n");
		return SC_ERROR_NOT_SUPPORTED;
	}
	if (se_num > 0)
	{
		sc_log(card->ctx, "restore security environment not supported.\n");
		return SC_ERROR_NOT_SUPPORTED;
	}
	sc_format_apdu(card, &apdu, SC_APDU_CASE_3_SHORT, 0x22, 0, 0);
	switch (env->operation)
	{
	case SC_SEC_OPERATION_DECIPHER:
		apdu.p1 = 0x41;
		apdu.p2 = 0xB8;
		break;
	case SC_SEC_OPERATION_SIGN:
		apdu.p1 = 0x41;
		apdu.p2 = 0xB6;
		break;
	case SC_SEC_OPERATION_UNWRAP:
		apdu.p1 = 0x41;
		apdu.p2 = 0xB8;
		break;
	case SC_SEC_OPERATION_WRAP:
		apdu.p1 = 0x81;
		apdu.p2 = 0xB8;
		break;
	case SC_SEC_OPERATION_ENCRYPT_SYM:
		apdu.p1 = 0x81;
		apdu.p2 = 0xB8;
		break;
	case SC_SEC_OPERATION_DECRYPT_SYM:
		apdu.p1 = 0x41;
		apdu.p2 = 0xB8;
		break;
	default:
		return SC_ERROR_INVALID_ARGUMENTS;
	}
	apdu.le = 0;
	p = sbuf;
	if (env->flags & SC_SEC_ENV_ALG_REF_PRESENT)
	{
		*p++ = 0x80;	
		*p++ = 0x01;
		*p++ = env->algorithm_ref & 0xFF;
	}
	if (env->flags & SC_SEC_ENV_FILE_REF_PRESENT)
	{
		*p++ = 0x81;
		*p++ = 2;
		memcpy(p, env->file_ref.value, 2);
		p += 2;
	}
	if (IS_SYMETRIC_CRYPT(env->operation)) {
		*p++ = 0x83;
		*p++ = 1;
		*p++ = 0;
	}
	if (env->flags & SC_SEC_ENV_KEY_REF_PRESENT && env->operation != SC_SEC_OPERATION_UNWRAP &&
			env->operation != SC_SEC_OPERATION_WRAP &&
			env->operation != SC_SEC_OPERATION_ENCRYPT_SYM &&
			env->operation != SC_SEC_OPERATION_DECRYPT_SYM) {
		*p++ = 0x84;
		*p++ = 1;
		*p++ = 0;
	}
	for (i = 0; i < SC_SEC_ENV_MAX_PARAMS; i++)
	    if (env->params[i].param_type == SC_SEC_ENV_PARAM_TARGET_FILE) {
			target_file = (sc_path_t*) env->params[i].value;
			if (env->params[i].value_len < sizeof(sc_path_t) || target_file->len != 2) {
				sc_log(card->ctx, "wrong length of target file reference.\n");
				return SC_ERROR_WRONG_LENGTH;
			}
			*p++ = 0x83;
			*p++ = 2;
			memcpy(p, target_file->value, 2);
			p+= 2;
			break;
	    }
	r = 0;
	if (env->operation == SC_SEC_OPERATION_UNWRAP || env->operation == SC_SEC_OPERATION_WRAP ||
			IS_SYMETRIC_CRYPT(env->operation)) {
		for (i = 0; i < SC_SEC_ENV_MAX_PARAMS; i++)
			if (env->params[i].param_type == SC_SEC_ENV_PARAM_IV) {
				r = 1;
				*p++ = 0x87;
				*p++ = (unsigned char) env->params[i].value_len;
				if (p + env->params[i].value_len >= sbuf + SC_MAX_APDU_BUFFER_SIZE) {
					sc_log(card->ctx, "IV too long.\n");
					return SC_ERROR_WRONG_LENGTH;
				}
				memcpy(p, env->params[i].value, env->params[i].value_len);
				p+=(unsigned char) env->params[i].value_len;
				break;
			}
	}
	if (IS_SYMETRIC_CRYPT(env->operation) && env->algorithm == SC_ALGORITHM_AES &&
			env->algorithm_flags == SC_ALGORITHM_AES_ECB && r == 0) {
		*p++ = 0x87;
		*p++ = 16;
		memset(p, 0, 16);
		p += 16;
	}
	r = p - sbuf;
	apdu.lc = r;
	apdu.datalen = r;
	apdu.data = sbuf;
	apdu.resplen = 0;
	if (apdu.datalen != 0)
	{
		r = sc_transmit_apdu(card, &apdu);
		if (r)
		{
			sc_log(card->ctx,
				"%s: APDU transmit failed", sc_strerror(r));
			goto err;
		}
		r = sc_check_sw(card, apdu.sw1, apdu.sw2);
		if (r)
		{
			sc_log(card->ctx,
				"%s: Card returned error", sc_strerror(r));
			goto err;
		}
	}
err:
	LOG_FUNC_RETURN(card->ctx, r);
}
static int myeid_set_security_env_ec(sc_card_t *card, const sc_security_env_t *env,
		int se_num)
{
	sc_apdu_t apdu;
	u8 sbuf[SC_MAX_APDU_BUFFER_SIZE];
	u8 *p;
	int r;
	assert(card != NULL && env != NULL);
	LOG_FUNC_CALLED(card->ctx);
	if (env->flags & SC_SEC_ENV_KEY_REF_SYMMETRIC)
	{
		sc_log(card->ctx, "symmetric keyref not supported.");
		return SC_ERROR_NOT_SUPPORTED;
	}
	if (se_num > 0)
	{
		sc_log(card->ctx, "restore security environment not supported.");
		return SC_ERROR_NOT_SUPPORTED;
	}
	sc_format_apdu(card, &apdu, SC_APDU_CASE_3_SHORT, 0x22, 0, 0);
	switch (env->operation)
	{
	case SC_SEC_OPERATION_DECIPHER:
		sc_log(card->ctx, "Decipher operation is not supported with EC keys.");
		return SC_ERROR_NOT_SUPPORTED;
		break;
	case SC_SEC_OPERATION_SIGN:
		apdu.p1 = 0x41;
		apdu.p2 = 0xB6;
		break;
	case SC_SEC_OPERATION_DERIVE:
		apdu.p1 = 0x41;
		apdu.p2 = 0xA4;
		break;
	default:
		return SC_ERROR_INVALID_ARGUMENTS;
	}
	apdu.le = 0;
	p = sbuf;
	if (env->flags & SC_SEC_ENV_ALG_REF_PRESENT)
	{
		*p++ = 0x80;	
		*p++ = 0x01;
		*p++ = env->algorithm_ref & 0xFF;
	}
	if (env->flags & SC_SEC_ENV_FILE_REF_PRESENT)
	{
		*p++ = 0x81;
		*p++ = 0x02;
		memcpy(p, env->file_ref.value, 2);
		p += 2;
	}
	if (env->flags & SC_SEC_ENV_KEY_REF_PRESENT)
	{
		*p++ = 0x84;
		*p++ = 1;
		*p++ = 0;
	}
	r = p - sbuf;
	apdu.lc = r;
	apdu.datalen = r;
	apdu.data = sbuf;
	apdu.resplen = 0;
	if (apdu.datalen != 0)
	{
		r = sc_transmit_apdu(card, &apdu);
		if (r)
		{
			sc_log(card->ctx,
				"%s: APDU transmit failed", sc_strerror(r));
			goto err;
		}
		r = sc_check_sw(card, apdu.sw1, apdu.sw2);
		if (r)
		{
			sc_log(card->ctx,
				"%s: Card returned error", sc_strerror(r));
			goto err;
		}
	}
err:
	LOG_FUNC_RETURN(card->ctx, r);
}
static int myeid_set_security_env(struct sc_card *card,
		const struct sc_security_env *env, int se_num)
{
	struct sc_context *ctx = card->ctx;
	myeid_private_data_t* priv;
	LOG_FUNC_CALLED(ctx);
	priv = (myeid_private_data_t*) card->drv_data;
	priv->sec_env = env;
	priv->algorithm_flags = env->algorithm_flags;
	priv->algorithm = env->algorithm;
	if (env->flags & SC_SEC_ENV_ALG_PRESENT)
	{
		sc_security_env_t tmp;
		tmp = *env;
		tmp.flags &= ~SC_SEC_ENV_ALG_PRESENT;
		tmp.flags |= SC_SEC_ENV_ALG_REF_PRESENT;
		if (tmp.algorithm == SC_ALGORITHM_RSA)
		{
			if (tmp.operation == SC_SEC_OPERATION_UNWRAP || tmp.operation == SC_SEC_OPERATION_WRAP)
			{
			    tmp.algorithm_ref = 0x0A;
			}
			else
			{
				tmp.algorithm_ref = 0x00;
				if (env->algorithm_flags & SC_ALGORITHM_RSA_PAD_PKCS1)
					tmp.algorithm_ref = 0x02;
				if (tmp.algorithm_flags & SC_ALGORITHM_RSA_HASH_SHA1)
					tmp.algorithm_ref |= 0x10;
			}
			return myeid_set_security_env_rsa(card, &tmp, se_num);
		}
		else if (tmp.algorithm == SC_ALGORITHM_EC)
		{
			tmp.algorithm_ref = 0x04;
			tmp.algorithm_flags = 0;
			return myeid_set_security_env_ec(card, &tmp, se_num);
		}
		else if (tmp.algorithm == SC_ALGORITHM_AES)
		{
			if (tmp.operation == SC_SEC_OPERATION_UNWRAP || tmp.operation == SC_SEC_OPERATION_WRAP)
			{
				tmp.algorithm_ref = 0x0A;
			}
			else
			{
				tmp.algorithm_ref = 0x00;
			}
			if ((tmp.algorithm_flags & SC_ALGORITHM_AES_CBC_PAD) == SC_ALGORITHM_AES_CBC_PAD)
				tmp.algorithm_ref |= 0x80;		
			if (IS_SYMETRIC_CRYPT(env->operation))
				tmp.algorithm_ref &= ~0x80; 
			return myeid_set_security_env_rsa(card, &tmp, se_num);
		}
		else
		{
			sc_log(ctx, "Unsupported algorithm.");
			return SC_ERROR_NOT_SUPPORTED;
		}
	}
	return myeid_set_security_env_rsa(card, env, se_num);
}
static int
myeid_convert_ec_signature(struct sc_context *ctx, size_t s_len, unsigned char *data, size_t datalen)
{
	unsigned char *buf;
	size_t buflen;
	int r;
	size_t len_size = 1;
	size_t sig_len = 0;
	assert(data && datalen && datalen > 3);
	if (*data != 0x30)
		return SC_ERROR_INVALID_DATA;
	if ((*(data + 1) & 0x80) == 0x80)
		len_size += *(data + 1) & 0x7F;
	if (len_size == 1)
	    sig_len = *(data + 1);
	else if (len_size == 2)
	    sig_len = *(data + 2);
	else if (len_size == 3)
	{
	    sig_len = *(data + 2) | (*data + 3) << 8;
	}
	else
	    return SC_ERROR_INVALID_DATA;
	if (*(data + 1 + len_size) != 0x02)		
	if (sig_len != (datalen - len_size - 1))	
	    return SC_ERROR_INVALID_DATA;
	buflen = (s_len + 7)/8*2;
	if (buflen > datalen)
		LOG_FUNC_RETURN(ctx, SC_ERROR_INVALID_DATA);
	buf = calloc(1, buflen);
	if (!buf)
		LOG_FUNC_RETURN(ctx, SC_ERROR_OUT_OF_MEMORY);
	r = sc_asn1_sig_value_sequence_to_rs(ctx, data, datalen, buf, buflen);
	if (r < 0) {
		free(buf);
		sc_log(ctx, "Failed to convert Sig-Value to the raw RS format");
		return r;
	}
	memmove(data, buf, buflen);
	free(buf);
	return buflen;
}
static int
myeid_compute_raw_2048_signature(struct sc_card *card, const u8 * data, size_t datalen,
		u8 * out, size_t outlen)
{
	int r;
	struct sc_context *ctx;
	struct myeid_private_data *priv;
	struct sc_apdu apdu;
	u8 rbuf[SC_MAX_APDU_BUFFER_SIZE];
	u8 sbuf[SC_MAX_APDU_BUFFER_SIZE];
	sc_security_env_t env;
	ctx = card->ctx;
	LOG_FUNC_CALLED(ctx);
	priv = (myeid_private_data_t *) card->drv_data;
	memcpy(&env, priv->sec_env, sizeof(sc_security_env_t));
	env.flags |= SC_SEC_ENV_ALG_REF_PRESENT;
	env.flags |= SC_SEC_ENV_FILE_REF_PRESENT;
	env.flags |= SC_SEC_ENV_KEY_REF_PRESENT;
	env.operation = SC_SEC_OPERATION_DECIPHER;
	myeid_set_security_env_rsa(card, &env, 0);
	sc_format_apdu(card, &apdu, SC_APDU_CASE_3_SHORT, 0x2A, 0x80, 0x86);
	apdu.resp = rbuf;
	apdu.resplen = sizeof(rbuf);
	apdu.le = 0;	
	sbuf[0] = 0x81;
	memcpy(sbuf + 1, data, datalen / 2);
	apdu.lc = datalen / 2 + 1;
	apdu.datalen = apdu.lc;
	apdu.data = sbuf;
	r = sc_transmit_apdu(card, &apdu);
	LOG_TEST_RET(card->ctx, r, "APDU transmit failed");
	if (apdu.sw1 == 0x90 && apdu.sw2 == 0x00) {
		sc_format_apdu(card, &apdu, SC_APDU_CASE_4_SHORT, 0x2A, 0x80, 0x86);
		apdu.resp = rbuf;
		apdu.resplen = sizeof(rbuf);
		apdu.le = datalen;
		sbuf[0] = 0x82;
		memcpy(sbuf + 1, data + datalen / 2, datalen / 2);
		apdu.lc = datalen / 2 + 1;
		apdu.datalen = apdu.lc;
		apdu.data = sbuf;
		r = sc_transmit_apdu(card, &apdu);
		LOG_TEST_RET(card->ctx, r, "APDU transmit failed");
		if (apdu.sw1 == 0x90 && apdu.sw2 == 0x00) {
			int len = apdu.resplen > outlen ? outlen : apdu.resplen;
			memcpy(out, apdu.resp, len);
			LOG_FUNC_RETURN(card->ctx, len);
		}
	}
	LOG_FUNC_RETURN(card->ctx, sc_check_sw(card, apdu.sw1, apdu.sw2));
}
static int
myeid_compute_signature(struct sc_card *card, const u8 * data, size_t datalen,
		u8 * out, size_t outlen)
{
	struct sc_context *ctx;
	struct sc_apdu apdu;
	u8 rbuf[MYEID_MAX_EXT_APDU_BUFFER_SIZE];
	u8 sbuf[MYEID_MAX_EXT_APDU_BUFFER_SIZE];
	struct myeid_private_data* priv;
	int r;
	size_t field_length = 0;
	size_t pad_chars = 0;
	assert(card != NULL && data != NULL && out != NULL);
	ctx = card->ctx;
	LOG_FUNC_CALLED(ctx);
	priv = (myeid_private_data_t*) card->drv_data;
	sc_log(ctx, "key type %i, key length %i", priv->sec_env->algorithm, priv->sec_env->algorithm_ref);
	if (priv->sec_env->algorithm == SC_ALGORITHM_EC ) {
	    field_length = priv->sec_env->algorithm_ref;
		if (datalen < (field_length + 7) / 8 ) {
			pad_chars = ((field_length + 7) / 8) - datalen;
			memset(sbuf, 0, pad_chars);
		}
	}
	if ((datalen + pad_chars) > sizeof(sbuf))
		LOG_FUNC_RETURN(ctx, SC_ERROR_INVALID_ARGUMENTS);
	if (priv->sec_env->algorithm == SC_ALGORITHM_RSA && datalen == 256 && !priv->cap_chaining)
		return myeid_compute_raw_2048_signature(card, data, datalen, out, outlen);
	sc_format_apdu(card, &apdu, SC_APDU_CASE_4_SHORT, 0x2A, 0x9E, 0x9A);
	apdu.flags |= SC_APDU_FLAGS_CHAINING;
	apdu.resp = rbuf;
	apdu.resplen = sizeof(rbuf);
	apdu.le = 256;
	memcpy(sbuf + pad_chars, data, datalen);
	apdu.lc = datalen + pad_chars;
	apdu.datalen = datalen + pad_chars;
	apdu.data = sbuf;
	r = sc_transmit_apdu(card, &apdu);
	LOG_TEST_RET(ctx, r, "APDU transmit failed");
	r = sc_check_sw(card, apdu.sw1, apdu.sw2);
	LOG_TEST_RET(ctx, r, "compute_signature failed");
	if (priv->sec_env->algorithm == SC_ALGORITHM_EC) {
		r = myeid_convert_ec_signature(ctx, priv->sec_env->algorithm_ref, apdu.resp, apdu.resplen);
		LOG_TEST_RET(ctx, r, "compute_signature convert signature failed");
		apdu.resplen = r;
	}
	if (apdu.resplen > outlen)
		LOG_FUNC_RETURN(ctx, SC_ERROR_BUFFER_TOO_SMALL);
	memcpy(out, apdu.resp, apdu.resplen);
	LOG_FUNC_RETURN(ctx, apdu.resplen);
}
int myeid_ecdh_derive(struct sc_card *card, const u8* pubkey, size_t pubkey_len, u8* out, size_t outlen)
{
	struct sc_apdu apdu;
	u8 sbuf[SC_MAX_APDU_BUFFER_SIZE];
	u8 rbuf[SC_MAX_APDU_BUFFER_SIZE];
	int r;
	size_t ext_len_bytes;
	sc_format_apdu(card, &apdu, SC_APDU_CASE_4_SHORT, 0x86, 0x00, 0x00);
	apdu.resp = rbuf;
	apdu.resplen = sizeof(rbuf);
	sbuf[0] = 0x7C;
	ext_len_bytes = 0;
	if (pubkey_len > 127)
	{
		sbuf[1] = 0x81;
		sbuf[2] = (u8) (pubkey_len + 3);
		sbuf[3] = 0x85;
		sbuf[4] = 0x81;
		sbuf[5] = (u8) (pubkey_len);
		ext_len_bytes = 2;
	}
	else
	{
		sbuf[1] = pubkey_len + 2;
		sbuf[2] = 0x85;
		sbuf[3] = pubkey_len;
	}
	memcpy(&sbuf[4 + ext_len_bytes], pubkey, pubkey_len);
	apdu.lc = pubkey_len + 4 + ext_len_bytes;
	apdu.le = pubkey_len / 2;
	apdu.datalen = apdu.lc;
	apdu.data = sbuf;
	r = sc_transmit_apdu(card, &apdu);
	LOG_TEST_RET(card->ctx, r, "APDU transmit failed.");
	r = sc_check_sw(card, apdu.sw1, apdu.sw2);
	LOG_TEST_RET(card->ctx, r, "ECDH operation failed - GENERAL AUTHENTICATE returned error.");
	if (outlen < apdu.resplen)
	{
		r = SC_ERROR_BUFFER_TOO_SMALL;
		LOG_TEST_RET(card->ctx, r, "Buffer too small to hold shared secret.");
	}
	memcpy(out, rbuf, apdu.resplen);
	LOG_FUNC_RETURN(card->ctx, apdu.resplen);
}
static int myeid_transmit_decipher_pi_split(struct sc_card *card, struct sc_apdu *apdu, u8 *sbuf)
{
	size_t crgram_len = apdu->lc - 1;
	size_t crgram_half = crgram_len / 2;
	size_t resplen = apdu->resplen;
	unsigned char *resp = apdu->resp;
	int r;
	LOG_FUNC_CALLED(card->ctx);
	apdu->cse = SC_APDU_CASE_3_SHORT;
	apdu->data = &sbuf[0];
	apdu->datalen = apdu->lc = crgram_half + 1;
	apdu->resp = 0;
	apdu->resplen = 0;
	apdu->le = 0;
	sbuf[0] = 0x81;			
	r = sc_transmit_apdu(card, apdu);
	LOG_TEST_RET(card->ctx, r, "APDU transmit failed");
	if (apdu->sw1 != 0x90 || apdu->sw2 != 0x00)
		return 0;
	apdu->cse = resplen ? SC_APDU_CASE_4_SHORT : SC_APDU_CASE_3_SHORT;
	apdu->data = &sbuf[crgram_half];
	apdu->datalen = apdu->lc = crgram_len - crgram_half + 1;
	apdu->resp = resp;
	apdu->resplen = resplen;
	apdu->le = resplen ? MIN(card->max_recv_size, crgram_len) : 0;
	sbuf[crgram_half] = 0x82;	
	r = sc_transmit_apdu(card, apdu);
	LOG_FUNC_RETURN(card->ctx, r);
}
static int myeid_transmit_decipher(struct sc_card *card, u8 p1, u8 p2,
		const u8 * crgram, size_t crgram_len, u8 * out, size_t outlen)
{
	myeid_private_data_t *priv = card->drv_data;
	struct sc_apdu apdu;
	u8 rbuf[SC_MAX_EXT_APDU_BUFFER_SIZE];
	u8 sbuf[SC_MAX_EXT_APDU_BUFFER_SIZE];
	int r;
	LOG_FUNC_CALLED(card->ctx);
	sc_format_apdu(card, &apdu, p1 ? SC_APDU_CASE_4_SHORT : SC_APDU_CASE_3_SHORT, 0x2A, p1, p2);
	if (p2 == 0x86) {
		if (crgram_len+1 > sizeof(sbuf))
			LOG_FUNC_RETURN(card->ctx, SC_ERROR_INVALID_ARGUMENTS);
		sbuf[0] = 0; 
		memcpy(sbuf + 1, crgram, crgram_len);
		apdu.data = sbuf;
		apdu.datalen = apdu.lc = crgram_len + 1;
	} else {
		apdu.data = crgram;
		apdu.datalen = apdu.lc = crgram_len;
	}
	if (p1 != 0x00) {
		apdu.resp = rbuf;
		apdu.resplen = sizeof(rbuf);
		apdu.le = MIN(card->max_recv_size, crgram_len);
	}
	if (p2 == 0x86 && crgram_len == 256 && priv && !priv->cap_chaining) {
		r = myeid_transmit_decipher_pi_split(card, &apdu, sbuf);
	} else {
		apdu.flags |= SC_APDU_FLAGS_CHAINING;
		r = sc_transmit_apdu(card, &apdu);
	}
	LOG_TEST_RET(card->ctx, r, "APDU transmit failed");
	r = sc_check_sw(card, apdu.sw1, apdu.sw2);
	LOG_TEST_RET(card->ctx, r, "DECIPHER returned error");
	if (out && outlen) {
		outlen = MIN(apdu.resplen, outlen);
		memcpy(out, apdu.resp, outlen);
	} else {
		outlen = 0;
	}
	LOG_FUNC_RETURN(card->ctx, outlen);
}
static int myeid_unwrap_key(struct sc_card *card, const u8 *crgram, size_t crgram_len)
{
	myeid_private_data_t* priv;
	u8 p2 = 0x86; 
	int r;
	if (card == NULL || crgram == NULL)
		return SC_ERROR_INVALID_ARGUMENTS;
	priv = card->drv_data;
	LOG_FUNC_CALLED(card->ctx);
	if (crgram_len > MYEID_MAX_RSA_KEY_LEN / 8)
		LOG_FUNC_RETURN(card->ctx, SC_ERROR_INVALID_ARGUMENTS);
	if (priv && priv->sec_env)
	{
		if (priv->sec_env->algorithm == SC_ALGORITHM_AES ||
			priv->sec_env->algorithm == SC_ALGORITHM_3DES ||
			priv->sec_env->algorithm == SC_ALGORITHM_DES)
				p2 = 0x84;
	}
	if (p2 == 0x84 && crgram_len > MYEID_MAX_APDU_DATA_LEN)
		LOG_TEST_RET(card->ctx, SC_ERROR_WRONG_LENGTH, "Unwrapping symmetric data longer that 255 bytes is not supported\n");
	r = myeid_transmit_decipher(card, 0x00, p2, crgram, crgram_len, 0, 0);
	LOG_FUNC_RETURN(card->ctx, r);
}
static int myeid_loadkey(sc_card_t *card, unsigned mode, u8* value, int value_len)
{
	myeid_private_data_t *priv = (myeid_private_data_t *) card->drv_data;
	sc_apdu_t apdu;
	u8 sbuf[MYEID_MAX_EXT_APDU_BUFFER_SIZE];
	int r;
	LOG_FUNC_CALLED(card->ctx);
	if (value_len == 0 || value == NULL)
		return 0;
	if (mode == LOAD_KEY_MODULUS && value_len == 256 && !priv->cap_chaining)
	{
		mode = 0x88;
		memset(&apdu, 0, sizeof(apdu));
		sc_format_apdu(card, &apdu, SC_APDU_CASE_3_SHORT, 0xDA, 0x01, mode);
		apdu.cla     = 0x00;
		apdu.data    = value;
		apdu.datalen = 128;
		apdu.lc	     = 128;
		r = sc_transmit_apdu(card, &apdu);
		LOG_TEST_RET(card->ctx, r, "APDU transmit failed");
		r = sc_check_sw(card, apdu.sw1, apdu.sw2);
		LOG_TEST_RET(card->ctx, r, "LOAD KEY returned error");
		mode = 0x89;
		value += 128;
		value_len -= 128;
	}
	else if ((mode & 0xff00) == 0 && mode != LOAD_KEY_PUBLIC_EXPONENT &&
		 value[0] != 0x00)
	{
		sbuf[0] = 0x0;
		memcpy(&sbuf[1], value, value_len);
		value = sbuf;
		value_len ++;
	}
	memset(&apdu, 0, sizeof(apdu));
	sc_format_apdu(card, &apdu, SC_APDU_CASE_3_SHORT, 0xDA, 0x01, mode & 0xFF);
	apdu.flags   = SC_APDU_FLAGS_CHAINING;
	apdu.cla     = 0x00;
	apdu.data    = value;
	apdu.datalen = value_len;
	apdu.lc	     = value_len;
	r = sc_transmit_apdu(card, &apdu);
	LOG_TEST_RET(card->ctx, r, "APDU transmit failed");
	r = sc_check_sw(card, apdu.sw1, apdu.sw2);
	LOG_FUNC_RETURN(card->ctx, r);
}
static int myeid_generate_store_key(struct sc_card *card,
	struct sc_cardctl_myeid_gen_store_key_info *data)
{
	struct	sc_apdu apdu;
	u8	sbuf[SC_MAX_APDU_BUFFER_SIZE];
	int	r=0,len;
	LOG_FUNC_CALLED(card->ctx);
	if (data->op_type == OP_TYPE_GENERATE)
	{
		len = 0;
		memset(&apdu, 0, sizeof(apdu));
		if(data->key_type == SC_CARDCTL_MYEID_KEY_RSA)
		{
		    sbuf[len++] = 0x30;
		    sbuf[len++] = 0x05;
		    sbuf[len++] = 0x81;
		    sbuf[len++] = data->pubexp_len;
		    memcpy(sbuf + len, data->pubexp, data->pubexp_len);
		    len += data->pubexp_len;
			sc_format_apdu(card, &apdu, SC_APDU_CASE_3_SHORT, 0x46, 0x00, 0x00);
			apdu.data    = sbuf;
		}
		else if(data->key_type == SC_CARDCTL_MYEID_KEY_EC) {
			sc_format_apdu(card, &apdu, SC_APDU_CASE_1, 0x46, 0x00, 0x00);
			apdu.data    = NULL;
			apdu.resp	 = sbuf;
			apdu.resplen = 0x00;
			apdu.le		 = 0x00;
		}
		apdu.cla     = 0x00;
		apdu.datalen = len;
		apdu.lc	     = len;
		r = sc_transmit_apdu(card, &apdu);
		LOG_TEST_RET(card->ctx, r, "APDU transmit failed");
		r = sc_check_sw(card, apdu.sw1, apdu.sw2);
		LOG_TEST_RET(card->ctx, r, "GENERATE_KEY returned error");
	}
	else
	{
		if(data->key_type == SC_CARDCTL_MYEID_KEY_RSA)
		{
			if((r=myeid_loadkey(card, LOAD_KEY_PRIME_P,
				data->primep, data->primep_len)) >= 0 &&
			(r=myeid_loadkey(card, LOAD_KEY_PRIME_Q,
				data->primeq, data->primeq_len)) >= 0 &&
			(r=myeid_loadkey(card, LOAD_KEY_DP1,
				data->dp1, data->dp1_len)) >= 0 &&
			(r=myeid_loadkey(card, LOAD_KEY_DQ1,
				data->dq1, data->dq1_len)) >= 0 &&
			(r=myeid_loadkey(card, LOAD_KEY_INVQ,
				data->invq, data->invq_len)) >= 0 &&
			(r=myeid_loadkey(card, LOAD_KEY_MODULUS,
				data->mod, data->key_len_bits)) >= 0 &&
			(r=myeid_loadkey(card, LOAD_KEY_PUBLIC_EXPONENT,
				data->pubexp, data->pubexp_len)) >= 0)
				LOG_FUNC_RETURN(card->ctx, r);
		}
		else if(data->key_type == SC_CARDCTL_MYEID_KEY_EC) {
			if((r = myeid_loadkey(card, LOAD_KEY_EC_PRIVATE, data->d,
					data->d_len)) >= 0 &&
				(r = myeid_loadkey(card, LOAD_KEY_EC_PUBLIC, data->ecpublic_point,
					data->ecpublic_point_len)) >= 0)
			LOG_FUNC_RETURN(card->ctx, r);
		}
		else if(data->key_type == SC_CARDCTL_MYEID_KEY_AES ||
			data->key_type == SC_CARDCTL_MYEID_KEY_DES) {
			if((r = myeid_loadkey(card, LOAD_KEY_SYMMETRIC, data->d,
					data->d_len)) >= 0)
			LOG_FUNC_RETURN(card->ctx, r);
		}
	}
	LOG_FUNC_RETURN(card->ctx, r);
}
static int myeid_get_card_caps(struct sc_card *card, myeid_card_caps_t* card_caps)
{
	sc_apdu_t apdu;
	int r;
	unsigned char rbuf[SC_MAX_APDU_BUFFER_SIZE];
	LOG_FUNC_CALLED(card->ctx);
	sc_format_apdu(card, &apdu, SC_APDU_CASE_2_SHORT, 0xca, 0x01, 0xAA);
	apdu.resp    = rbuf;
	apdu.resplen = sizeof(myeid_card_caps_t);
	apdu.le      = sizeof(myeid_card_caps_t);
	r = sc_transmit_apdu(card, &apdu);
	LOG_TEST_RET(card->ctx, r,  "APDU transmit failed");
	if (apdu.sw1 != 0x90 || apdu.sw2 != 0x00)
		return SC_ERROR_INTERNAL;
	if (apdu.resplen < 11) {
		sc_log(card->ctx, "Unexpected response to GET DATA (MyEIC card capabilities)");
		return SC_ERROR_INTERNAL;
	}
	card_caps->card_caps_ver = rbuf[0];
	card_caps->card_supported_features = (unsigned short) rbuf[1] << 8 | rbuf[2];
	card_caps->max_rsa_key_length = (unsigned short) rbuf[3] << 8 | rbuf[4];
	card_caps->max_des_key_length = (unsigned short) rbuf[5] << 8 | rbuf[6];
	card_caps->max_aes_key_length = (unsigned short) rbuf[7] << 8 | rbuf[8];
	card_caps->max_ecc_key_length = (unsigned short) rbuf[9] << 8 | rbuf[10];
	LOG_FUNC_RETURN(card->ctx, r);
}
static int
myeid_enc_dec_sym(struct sc_card *card, const u8 *data, size_t datalen,
		u8 *out, size_t *outlen, int decipher)
{
	struct sc_context *ctx;
	struct sc_apdu apdu;
	u8 rbuf[SC_MAX_APDU_BUFFER_SIZE];
	u8 sbuf[SC_MAX_APDU_BUFFER_SIZE];
	u8 *sdata;
	int r, padding = 0, cbc = 0;
	size_t block_size;
	size_t len, rest_len;
	size_t return_len = 0;
	size_t max_apdu_datalen;
	size_t apdu_datalen;
	assert(card != NULL);
	ctx = card->ctx;
	LOG_FUNC_CALLED(ctx);
	myeid_private_data_t *priv;
	priv = (myeid_private_data_t *)card->drv_data;
	max_apdu_datalen = sc_get_max_send_size(card);
	if (max_apdu_datalen > sc_get_max_recv_size(card))
		max_apdu_datalen = sc_get_max_recv_size(card);
	if (max_apdu_datalen > SC_MAX_APDU_BUFFER_SIZE)
		max_apdu_datalen = SC_MAX_APDU_BUFFER_SIZE;
	sc_log(ctx, "algorithm %d algorithm_flags %x", priv->algorithm, priv->algorithm_flags);
	if (outlen == NULL) {
		sc_log(ctx, "%s (symmetric key) initialized", decipher ? "C_DecryptInit" : "C_EncryptInit");
		priv->sym_crypt_buffer_len = 0;
		priv->sym_plain_buffer_len = 0;
		return SC_SUCCESS;
	}
	switch (priv->algorithm) {
	case SC_ALGORITHM_AES:
		block_size = 16;
		if (priv->algorithm_flags & SC_ALGORITHM_AES_ECB) {
			padding = 0;
			cbc = 0;
		} else if (priv->algorithm_flags & SC_ALGORITHM_AES_CBC) {
			padding = 0;
			cbc = 1;
		} else if (priv->algorithm_flags & SC_ALGORITHM_AES_CBC_PAD) {
			padding = 1;
			cbc = 1;
		}
		break;
	default:
		LOG_FUNC_RETURN(ctx, SC_ERROR_NOT_SUPPORTED);
	}
	if (cbc)
		max_apdu_datalen -= max_apdu_datalen % block_size;
	else
		max_apdu_datalen = block_size;
	rest_len = priv->sym_crypt_buffer_len;
	if (data == NULL) {
		if (datalen != 0)
			LOG_FUNC_RETURN(ctx, SC_ERROR_WRONG_LENGTH);
		if (decipher) {
			if (priv->sym_plain_buffer_len != block_size)
				LOG_FUNC_RETURN(ctx, SC_ERROR_WRONG_LENGTH);
			if (rest_len)
				LOG_FUNC_RETURN(ctx, SC_ERROR_WRONG_LENGTH);
			return_len = block_size;
			if (padding) {
				uint8_t i, pad_byte = *(priv->sym_plain_buffer + block_size - 1);
				sc_log(ctx, "Found padding byte %02x", pad_byte);
				if (pad_byte == 0 || pad_byte > block_size)
					LOG_FUNC_RETURN(ctx, SC_ERROR_WRONG_PADDING);
				sdata = priv->sym_plain_buffer + block_size - pad_byte;
				for (i = 0; i < pad_byte; i++)
					if (sdata[i] != pad_byte)
						LOG_FUNC_RETURN(ctx, SC_ERROR_WRONG_PADDING);
				return_len = block_size - pad_byte;
			}
			*outlen = return_len;
			if (out == NULL)
				LOG_FUNC_RETURN(ctx, SC_SUCCESS);
			if (return_len > *outlen)
				LOG_FUNC_RETURN(ctx, SC_ERROR_BUFFER_TOO_SMALL);
			memcpy(out, priv->sym_plain_buffer, return_len);
			sc_log(ctx, "C_DecryptFinal %zu bytes", *outlen);
			return SC_SUCCESS;
		} else {
			if (padding) {
				uint8_t pad_byte = block_size - rest_len;
				sc_log(ctx, "Generating padding, padding byte: %d", pad_byte);
				sdata = priv->sym_crypt_buffer + rest_len;
				memset(sdata, pad_byte, pad_byte);
				rest_len = block_size;
			} else if (rest_len) {
				LOG_FUNC_RETURN(ctx, SC_ERROR_WRONG_LENGTH);
			}
		}
	}
	len = datalen + rest_len;
	sc_log(ctx, "datalen=%zu rest_len=%zu len=%zu outlen=%zu", datalen, rest_len, len, *outlen);
	len -= (len % block_size);
	*outlen = len;
	if (out == NULL)
		LOG_FUNC_RETURN(ctx, SC_SUCCESS);
	if (*outlen < len)
		LOG_FUNC_RETURN(ctx, SC_ERROR_BUFFER_TOO_SMALL);
	while (len >= block_size) {
		if (!decipher)
			sc_format_apdu(card, &apdu, SC_APDU_CASE_4_SHORT, 0x2A, 0x84, 0x80);
		else
			sc_format_apdu(card, &apdu, SC_APDU_CASE_4_SHORT, 0x2A, 0x80, 0x84);
		apdu.cla = 0;
		if (len > max_apdu_datalen)
			apdu_datalen = max_apdu_datalen;
		else
			apdu_datalen = len;
		if (cbc)
			apdu.cla = 0x10;
		len -= apdu_datalen;
		sdata = sbuf;
		apdu.le = apdu_datalen;
		apdu.lc = apdu_datalen;
		apdu.datalen = apdu_datalen;
		apdu.data = sbuf;
		apdu.resplen = sizeof(rbuf);
		apdu.resp = rbuf;
		if (rest_len) {
			memcpy(sbuf, priv->sym_crypt_buffer, rest_len);
			sdata += rest_len;
			apdu_datalen -= rest_len;
			priv->sym_crypt_buffer_len = 0;
			rest_len = 0;
		}
		if (data) {
			memcpy(sdata, data, apdu_datalen);
			data += apdu_datalen;
			datalen -= apdu_datalen;
		}
		r = sc_transmit_apdu(card, &apdu);
		LOG_TEST_RET(ctx, r, "APDU transmit failed");
		r = sc_check_sw(card, apdu.sw1, apdu.sw2);
		LOG_TEST_RET(ctx, r, "decrypt_sym/encrypt_sym failed");
		if (apdu.resplen != apdu.datalen)
			LOG_FUNC_RETURN(ctx, SC_ERROR_WRONG_LENGTH);
		memcpy(out, apdu.resp, apdu.resplen);
		out += apdu.resplen;
		return_len += apdu.resplen;
	}
	if (decipher) {
		uint8_t tmp_buf[16];
		if (return_len >= block_size) {
			memcpy(tmp_buf, out - block_size, block_size);
			if (priv->sym_plain_buffer_len) {
				sc_log(ctx, "inserting block from previous decrypt");
				memmove(out - return_len + block_size, out - return_len, return_len - block_size);
				memcpy(out - return_len, priv->sym_plain_buffer, block_size);
			} else
				return_len -= block_size;
			memcpy(priv->sym_plain_buffer, tmp_buf, block_size);
			priv->sym_plain_buffer_len = block_size;
		} else
			priv->sym_plain_buffer_len = 0;
	}
	priv->sym_crypt_buffer_len = datalen;
	sc_log(ctx, "rest data len = %zu", datalen);
	if (data)
		memcpy(priv->sym_crypt_buffer, data, datalen);
	sc_log(ctx, "return data len = %zu", return_len);
	*outlen = return_len;
	return SC_SUCCESS;
}
static struct sc_card_driver * sc_get_driver(void)
{
	struct sc_card_driver *iso_drv = sc_get_iso7816_driver();
	if (iso_ops == NULL)
		iso_ops = iso_drv->ops;
	myeid_ops			= *iso_drv->ops;
	myeid_ops.match_card		= myeid_match_card;
	myeid_ops.init			= myeid_init;
	myeid_ops.finish		= myeid_finish;
	myeid_ops.read_record		= NULL;
	myeid_ops.write_record		= NULL;
	myeid_ops.append_record		= NULL;
	myeid_ops.update_record		= NULL;
	myeid_ops.select_file		= myeid_select_file;
	myeid_ops.get_response		= iso_ops->get_response;
	myeid_ops.logout		= myeid_logout;
	myeid_ops.create_file		= myeid_create_file;
	myeid_ops.delete_file		= myeid_delete_file;
	myeid_ops.list_files		= myeid_list_files;
	myeid_ops.set_security_env	= myeid_set_security_env;
	myeid_ops.compute_signature	= myeid_compute_signature;
	myeid_ops.decipher		= myeid_decipher;
	myeid_ops.process_fci		= myeid_process_fci;
	myeid_ops.card_ctl		= myeid_card_ctl;
	myeid_ops.pin_cmd		= myeid_pin_cmd;
	myeid_ops.wrap			= myeid_wrap_key;
	myeid_ops.unwrap		= myeid_unwrap_key;
	myeid_ops.encrypt_sym		= myeid_encrypt_sym;
	myeid_ops.decrypt_sym		= myeid_decrypt_sym;
	return &myeid_drv;
}
